module talk

//  ********************************************************************************
//  Clean tutorial example program.
//  
//  This program creates two interactive processes. Each process opens a window in 
//	which the user can type text. Text that has been typed in one window is being 
//	sent to the other, and vice versa.
//  ********************************************************************************

import  StdEnv, StdIO

::  Message
    =   NewLine String
    |   Quit
::  NoState
    =   NoState

Start :: *World -> *World
Start world
    #   (a,world)   = openRId world
    #   (b,world)   = openRId world
    =   startProcesses
    		(ProcessGroup NoState (ListCS [talk "A" a b,talk "B" b a])
    		) world
where
    talk :: String (RId Message) (RId Message) -> MDIProcess .p
    talk name me you
        =   MDIProcess NoState [openTalkWindow name me you] [ProcessNoWindowMenu]

openTalkWindow :: String (RId Message) (RId Message) (PSt .l .p) -> PSt .l .p
openTalkWindow name me you ps
    #   menu        = Menu ("Talk "+++name)
                        (   MenuItem "Quit" [   MenuShortKey 'q'
                                            ,   MenuFunction (noLS quit)
                                            ]
                        )   []
    #   (error,ps)  = openMenu NoState menu ps
    |   error<>NoError
        =   abort "talk could not open menu."
    #   (wId,  ps)  = accPIO openId ps
    #   (inId, ps)  = accPIO openId ps
    #   (outId,ps)  = accPIO openId ps
    #   wdef        = Dialog ("Talk "+++name) 
                        (   EditControl "" (hmm 50.0) 5
                                    [   ControlId       inId
                                    ,   ControlKeyboard inputfilter Able
                                    					(noLS1 (input wId inId you))
                                    ]
                        :+: EditControl "" (hmm 50.0) 5
                                    [   ControlId       outId
                                    ,   ControlPos      (BelowPrev,zero)
                                    ,   ControlSelectState Unable
                                    ]
                        )
                        [   WindowId    wId
                        ]
    #   (error,ps)  = openDialog undef wdef ps
    |   error<>NoError
        =   abort "talk could not open window."
    #   rdef        = Receiver me (noLS1 (receive wId outId)) []
    #   (error,ps)  = openReceiver NoState rdef ps
    |   error<>NoError
        =   abort "talk could not open receiver"
    |   otherwise
        =   ps
where
    inputfilter :: KeyboardState -> Bool
    inputfilter keystate
        =   getKeyboardStateKeyState keystate<>KeyUp
    
    input :: Id Id (RId Message) KeyboardState (PSt .l .p) -> PSt .l .p
    input wId inId you _ ps
        #   (Just window,ps) = accPIO (getWindow wId) ps
            text             = fromJust (snd (hd (getControlTexts [inId] window)))
        =   snd (asyncSend you (NewLine text) ps)
        
    receive :: Id Id Message (PSt .l .p) -> PSt .l .p
    receive wId outId (NewLine text) ps
        =   appPIO (setWindow wId [ setControlTexts      [(outId,text)]
                                  , setEditControlCursor outId (size text)
                                  ])  ps
    receive _ _ Quit ps
        =   closeProcess ps
    
    quit :: (PSt .l .p) -> PSt .l .p
    quit ps
        =   closeProcess (snd (asyncSend you Quit ps))
